<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       http://github.com/goodshuffle/gspro-gallery
 * @since      0.1.0
 *
 * @package    Gspro_Gallery
 * @subpackage Gspro_Gallery/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      0.1.0
 * @package    Gspro_Gallery
 * @subpackage Gspro_Gallery/includes
 * @author     Jon Morton <jon@goodshuffle.com>
 */
class Gspro_Gallery
{

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    0.1.0
	 * @access   protected
	 * @var      Gspro_Gallery_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    0.1.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    0.1.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    0.1.0
	 */
	public function __construct()
	{
		if (defined('GSPRO_GALLERY_VERSION')) {
			$this->version = GSPRO_GALLERY_VERSION;
		} else {
			$this->version = '0.5.6';
		}
		$this->plugin_name = 'gspro-gallery';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_rewrite_rules();
		$this->define_admin_hooks();
		$this->define_public_hooks();
		$this->define_shortcodes();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Gspro_Gallery_Loader. Orchestrates the hooks of the plugin.
	 * - Gspro_Gallery_i18n. Defines internationalization functionality.
	 * - Gspro_Gallery_Admin. Defines all hooks for the admin area.
	 * - Gspro_Gallery_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    0.1.0
	 * @access   private
	 */
	private function load_dependencies()
	{

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-gspro-gallery-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-gspro-gallery-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-gspro-gallery-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'public/class-gspro-gallery-public.php';

		/**
		 * The class responsible for defining how shortcodes are produced.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-gspro-gallery-shortcodes.php';

		/**
		 * The class responsible for generating rewrite rules for the item detail page.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-gspro-gallery-rewrites.php';


		$this->loader = new Gspro_Gallery_Loader();
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Gspro_Gallery_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    0.1.0
	 * @access   private
	 */
	private function set_locale()
	{

		$plugin_i18n = new Gspro_Gallery_i18n();

		$this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    0.1.0
	 * @access   private
	 */
	private function define_admin_hooks()
	{
		$plugin_admin = new Gspro_Gallery_Admin($this->get_plugin_name(), $this->get_version());

		$this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
		$this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');
		$this->loader->add_action('admin_menu', $plugin_admin, 'add_plugin_admin_menu');
		$this->loader->add_action('admin_query_filter', $plugin_admin, 'add_query_params');


		$plugin_basename = plugin_basename(plugin_dir_path(__DIR__) . $this->plugin_name . '.php');
		$this->loader->add_filter('plugin_action_links_' . $plugin_basename, $plugin_admin, 'add_action_links');
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    0.1.0
	 * @access   private
	 */
	private function define_public_hooks()
	{
		$active = get_option('gspro_on', false);
		if ($active) {
			$plugin_public = new Gspro_Gallery_Public($this->get_plugin_name(), $this->get_version());
			$this->loader->add_action('wp_head', $plugin_public, 'enqueue_meta');
			$this->loader->add_action('wp_head', $plugin_public, 'enqueue_modern_webcomponents');
			$this->loader->add_action('wp_head', $plugin_public, 'enqueue_legacy_webcomponents');
			$this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_styles');
			$this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_scripts');
			$this->loader->add_action('template_redirect', $plugin_public, 'list_inject');
		}
	}

	/**
	 * Register gspro-* shortcodes.
	 * 
	 * Creating an instance of the shortcode class will:
	 * 1. Register each shortcode (obviously).
	 * 2. Cause related scripts and styles to be enqueued.
	 * 
	 */
	private function define_shortcodes()
	{
		// TBD: Pass vendor configuration to limit the scope of retrieved data
		//      to a specific vendor.
		$shortcodes = new Gspro_Gallery_Shortcodes();
	}

	private function define_rewrite_rules()
	{
		$rewrite_rules = new Gspro_Gallery_Rewrites();
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    0.1.0
	 */
	public function run()
	{
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     0.1.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name()
	{
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     0.1.0
	 * @return    Gspro_Gallery_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader()
	{
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     0.1.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version()
	{
		return $this->version;
	}
}
