<?php
/**
 * Custom template tags for this theme
 *
 * @package ComRepair
 */

if ( ! function_exists( 'comrepair_site_branding_text' ) ) :
	/**
	 * Display Site Title and Tagline.
	 */
	function comrepair_site_branding_text() {
		?>
		<div class="site-branding-text">
			<?php if ( is_front_page() ) : ?>
				<h1 class="site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></h1>
			<?php else : ?>
				<p class="site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p>
			<?php endif; ?>
			<?php if ( '' !== ( $description = get_bloginfo( 'description', 'display' ) ) ) : ?>
				<p class="site-description"><?php echo esc_html( $description ); ?></p>
			<?php endif; ?>
		</div><!-- .site-branding-text -->
		<?php
	}
endif;

if ( ! function_exists( 'comrepair_the_page_title' ) ) :
	/**
	 * Display the current page or post title.
	 */
	function comrepair_the_page_title() {
		// 404 page
		if ( is_404() ) {
			esc_html_e( 'Oops! That page can&rsquo;t be found.', 'comrepair' );
			return;
		}
		// Search pages
		if ( is_search() ) {
			if ( have_posts() ) {
				/* translators: Search results page title */
				esc_html_e( 'Search', 'comrepair' );
			} else {
				esc_html_e( 'Nothing found', 'comrepair' );
			}
			return;
		}
		// Single posts
		if ( is_singular( 'post' ) ) {
			echo get_the_title( get_option( 'page_for_posts', true ) );
			return;
		}
		// WooCommerce pages
		if ( comrepair_is_woocommerce_activated() && is_woocommerce() ) {
			woocommerce_page_title();
			return;
		}
		// Archive pages
		if ( is_archive() ) {
			the_archive_title();
			return;
		}
		// By default show the title
		single_post_title();
	}
endif;

if ( ! function_exists( 'comrepair_breadcrumbs' ) ) :
	/**
	 * Display the breadcrumbs trail.
	 *
	 * Depends on Breadcrumb NavXT plugin being installed.
	 */
	function comrepair_breadcrumbs() {
		// Check if the Breadcrumb NavXT plugin is installed.
		if ( ! function_exists( 'bcn_display' ) ) {
			return;
		}
		// Check if really should show the breadcrumbs.
		if ( ! is_customize_preview() && 'hide' === comrepair_get_setting( 'show_breadcrumbs' ) ) {
			return;
		}
		?>
		<div class="breadcrumbs" typeof="BreadcrumbList" vocab="http://schema.org/">
			<?php bcn_display(); ?>
		</div>
		<?php
	}
endif;

if ( ! function_exists( 'comrepair_header_cart' ) ) :
	/**
	 * Displays the header cart.
	 *
	 * Depends on WooCommerce plugin being installed.
	 */
	function comrepair_header_cart() {
		if ( ! comrepair_is_woocommerce_activated() ) {
			return;
		}
		?>
		<ul id="site-header-cart" class="site-header-cart menu">
			<li <?php if ( is_cart() ) { echo 'class="current-menu-item"'; } ?>>
				<?php comrepair_cart_link(); ?>
			</li>
			<li>
				<?php the_widget( 'WC_Widget_Cart', 'title=' ); ?>
			</li>
		</ul>
		<?php
	}
endif;

if ( ! function_exists( 'comrepair_cart_link' ) ) :
	/**
	 * Displays the cart link.
	 *
	 * Depends on WooCommerce plugin being installed.
	 */
	function comrepair_cart_link() {
		?>
		<a class="cart-contents" href="<?php echo esc_url( wc_get_cart_url() ); ?>" title="<?php esc_attr_e( 'View your shopping cart', 'comrepair' ); ?>">
			<?php if ( $count = WC()->cart->get_cart_contents_count() ) : ?>
				<span class="count"><?php echo esc_html( $count ); ?></span>
			<?php endif; ?>
		</a>
		<?php
	}
endif;

if ( ! function_exists( 'comrepair_pagination' ) ) :
	/**
	 * Displays the paging navigation on archive pages.
	 */
	function comrepair_pagination() {
		the_posts_pagination( array(
			'prev_text'          => '<i class="typcn typcn-arrow-left"></i><span class="screen-reader-text">' . esc_html__( 'Previous page', 'comrepair' ) . '</span>',
			'next_text'          => '<i class="typcn typcn-arrow-right"></i><span class="screen-reader-text">' . esc_html__( 'Next page', 'comrepair' ) . '</span>',
			'before_page_number' => '<span class="meta-nav screen-reader-text">' . esc_html__( 'Page', 'comrepair' ) . ' </span>',
		) );
	}
endif;

if ( ! function_exists( 'comrepair_post_date' ) ) :
	/**
	 * Displays the post date for post preview.
	 */
	function comrepair_post_date( $class = '' ) {
		?>

		<a href="<?php the_permalink(); ?>" rel="bookmark">
			<div class="<?php echo esc_attr( $class ); ?> post-date post-date--circle">
				<time datetime="<?php echo esc_attr( get_the_time( 'c' ) ); ?>">
					<span class="post-date-day"><?php echo esc_html( get_the_time( 'd' ) ); ?></span> <span class="post-date-month"><?php echo esc_html( get_the_time( 'M' ) ); ?></span>
				</time>
			</div>
		</a>

		<?php
	}
endif;

if ( ! function_exists( 'comrepair_post_share_links' ) ) :
	/**
	 * Displays a list of share links for the post.
	 */
	function comrepair_post_share_links() {
		$title = get_the_title();
		$permalink = get_the_permalink();
		$facebook_share_url = sprintf( 'https://www.facebook.com/sharer/sharer.php?u=%1$s&t=%2$s', rawurlencode( $permalink ), rawurlencode( $title ) );
		$twitter_share_url = sprintf( 'https://twitter.com/home?status=%s', rawurlencode( $title . ' - ' . $permalink ) );
		$google_plus_share_url = sprintf( 'https://plus.google.com/share?url=%s', rawurlencode( $permalink ) );
		$linkedin_share_url = sprintf( 'https://www.linkedin.com/shareArticle?mini=true&url=%1$s&title=%2$s&summary=%3$s', rawurlencode( $permalink ), rawurlencode( $title ), rawurlencode( get_the_excerpt() ) );
		?>

		<a href="<?php echo esc_url( $facebook_share_url ); ?>" title="<?php esc_attr_e( 'Share on Facebook', 'comrepair' ); ?>"></a>
		<a href="<?php echo esc_url( $twitter_share_url ); ?>" title="<?php esc_attr_e( 'Share on Twitter', 'comrepair' ); ?>"></a>
		<a href="<?php echo esc_url( $google_plus_share_url ); ?>" title="<?php esc_attr_e( 'Share on Google+', 'comrepair' ); ?>"></a>
		<a href="<?php echo esc_url( $linkedin_share_url ); ?>" title="<?php esc_attr_e( 'Share on LinkedIn', 'comrepair' ); ?>"></a>

		<?php
	}
endif;

if ( ! function_exists( 'comrepair_render_comment' ) ) :
	/**
	 * Displays a comment.
	 */
	function comrepair_render_comment( $comment, $args, $depth ) {
		?>

		<li id="comment-<?php comment_ID(); ?>" <?php comment_class( $comment->has_children ? 'parent' : '', $comment ); ?>>
			<article id="div-comment-<?php comment_ID(); ?>" class="comment-body">

				<footer class="comment-meta-wrap">

					<div class="comment-author vcard">
						<?php
						if ( 0 !== $args['avatar_size'] ) {
							echo get_avatar( $comment, $args['avatar_size'], '', '', array( 'class' => 'comment-author-avatar' ) );
						}
						echo '<span class="comment-author-name fn">' . get_comment_author_link( $comment ) . '</span>';
						if ( comrepair_is_post_author( $comment->user_id ) ) {
							echo '<span class="comment-author-label">' . esc_html_x( 'Author', 'comment author label', 'comrepair' ) . '</span>';
						}
						?>
					</div><!-- .comment-author -->

					<div class="comment-metadata">

						<span class="comment-date">
							<a href="<?php echo esc_url( get_comment_link( $comment, $args ) ); ?>">
								<time datetime="<?php comment_time( 'c' ); ?>">
									<?php echo esc_html( comrepair_to_relative_date( $comment->comment_date ) ); ?>
								</time>
							</a>
						</span>

						<div class="comment-actions-wrap">
							<?php
							edit_comment_link( esc_html__( 'Edit', 'comrepair' ), '<span class="comment-edit-link-wrap">', '</span>' );

							comment_reply_link( array_merge( $args, array(
								'add_below' => 'div-comment',
								'depth'     => $depth,
								'before'    => '<span class="comment-reply-link-wrap">',
								'after'     => '</span>',
							) ) );
							?>
						</div>

					</div><!-- .comment-metadata -->

					<?php if ( '0' === $comment->comment_approved ) : ?>
						<p class="comment-awaiting-moderation"><?php esc_html_e( 'Your comment is awaiting moderation.', 'comrepair' ); ?></p>
					<?php endif; ?>

				</footer><!-- .comment-meta -->

				<div class="comment-content">
					<?php comment_text(); ?>
				</div><!-- .comment-content -->

			</article><!-- .comment-body -->

		<?php
	}
endif;

if ( ! function_exists( 'comrepair_header_info_block' ) ) :
	/**
	 * Displays a header info block with text and icon that were set via the Customizer.
	 */
	function comrepair_header_info_block( $id ) {
		if ( $text = get_theme_mod( 'header_info_block_text_' . $id ) ) {
			$icon_class = get_theme_mod( 'header_info_block_icon_class_' . $id );
			$icon_style = get_theme_mod( 'header_info_block_icon_style_' . $id );
			?>

			<div class="header-info-block js-header-info-block-<?php echo esc_attr( $id ); ?>">

				<?php if ( $icon_class || is_customize_preview() ) : ?>
					<div class="header-info-block-icon <?php if ( $icon_style ) { echo 'header-info-block-icon--' . esc_attr( $icon_style ); } ?>">
						<span class="<?php echo esc_attr( $icon_class ); ?>"></span>
					</div><!-- .header-info-block-icon -->
				<?php endif; ?>

				<div class="header-info-block-text">
					<?php echo wp_kses_post( html_entity_decode( $text ) ); ?>
				</div><!-- .header-info-block-text -->

			</div><!-- .header-info-block -->

			<?php
		} elseif ( is_customize_preview() ) {
			echo '<div class="header-info-block-placeholder js-header-info-block-' . esc_attr( $id ) . '"></div>';
		}
	}
endif;

if ( ! function_exists( 'comrepair_shop_sidebar' ) ) :
	/**
	 * Displays Shop sidebar.
	 */
	function comrepair_shop_sidebar() {
		if ( ! is_active_sidebar( 'shop-sidebar-1' ) ) {
			return;
		}
		?>

		<div class="sidebar-wrap">
			<aside id="secondary" class="sidebar-content">
				<?php dynamic_sidebar( 'shop-sidebar-1' ); ?>
			</aside><!-- #secondary -->
		</div><!-- .sidebar-wrap -->

		<?php
	}
endif;
